%% Customizing Simulink Graphical User Interface
% This demonstration shows how you can customize the Simulink graphical user interface (GUI). 
%
% We will demonstrate four capabilities:
%
% * Reordering Libraries in Simulink Library Browser.
% * Disabling and Hiding Libraries in Simulink Library Browser.
% * Customizing Simulink Menus.
% * Disabling and Hiding Options in Configuration Paramaters Dialog.
%
% To make the customizations we use different versions of the |sl_customization.m| file. 
% In this demo, we use three different versions, each covering one of 
% the capabilities in the above list.
%
% To make each customization effective, we add the appropriate 
% version of sl_customization.m to the MATLAB path and then update the Simulink
% user interface with the settings specified in that file.

%% Reordering Libraries in Simulink Library Browser
% In this section we show how you can reorder libraries in your
% Simulink Library Browser. Our reordering will place the Simulink Extras library above
% the Simulink library. 
% On opening Simulink library browser, you can see that the Simulink
% library is placed above relative to the Simulink Extras library.

simulink('open');

%%
% <<sl_custom_demo_01.png>>

%%
% Now let us look at |sl_customization.m| file that we will use to make the
% changes to the order of libraries.
%
% In this demo, the |sl_custom_order.m| file is copied into the current
% directory as |sl_customization.m|. Since the current directory is always
% on the MATLAB path, update of the Simulink GUI as specified by |sl_customization.m| 
% will be applied automatically.
%
% However, it is recommended that when customizing Simulink GUI for your project, you add
% the directory that contains the |sl_customization.m| file to the MATLAB
% path using the following command: 
%
% |addpath(dir,'-end');|
%
% where |dir| is the directory that constains the |sl_customization.m| file

copyfile('sl_custom_order.m', 'sl_customization.m', 'f');
type sl_customization.m

%%
% |sl_refresh_customizations| function applies the changes to the Simulink GUI settings.
% There is a change in the order of libraries in the Simulink library
% browser.
%
% Note that the name of the libraries in the Simulink browser can be
% obtained by using the following command:
%
% |findblib|
%
% and then clicking on the library. The name of the libary can be obtained
% from the top left corner. 

sl_refresh_customizations;

%%
% <<sl_custom_demo_02.png>>

%% 
% If the directory containing |sl_customization.m| to MATLAB path, you can revert to
% the original GUI settings by removing this directory from the path using the following command: 
%
% |rmpath(pwd);| 
%
% In this demo, we simply delete |sl_customization.m| from the current directory. 
% The Simulink library browser reverts to the original GUI settings after the the 
% |sl_refresh_customizations| function is executed. 

delete('sl_customization.m');
sl_refresh_customizations;
simulink('close');

%% Disabling and Hiding Libraries in Simulink Library Browser
% In this section we show how you can disable and/or hide libraries in
% the Simulink library browser. We hide the Continuous library and
% disable the Discrete library. Let us look at |sl_customization.m| file
% that brings about these changes.

simulink('open');
copyfile('sl_custom_filter.m', 'sl_customization.m', 'f');
type sl_customization.m

%%
% Observe the changes in Simulink library browser - Continuous library is hidden and Discrete
% library is disabled.

sl_refresh_customizations;

%%
% <<sl_custom_demo_03.png>>

%% 
% Simulink reverts to the original GUI settings after the |sl_customization.m| 
% is removed from the MATLAB path and the |sl_refresh_customizations|
% function is executed.

delete('sl_customization.m');
sl_refresh_customizations;
simulink('close');

%%  Customizing Simulink Menus
% In this section we show how you can customize Simulink menus. Let us look 
% at |sl_customization.m| file that brings about these changes.

copyfile('sl_custom_menu.m', 'sl_customization.m','f');
type sl_customization.m

%%
% Clicking on the *Tools* menu of F14 model(F-14 Longitudinal Flight Control demo) shows the changes. 
% There are new items on this menu that were defined in |sl_customization.m| file.
% If *Item One* menu item is clicked, it will execute the callback function and print 
% a message that identifies the callback for *Item One*. Since *Item Two* is made up of three 
% *Item One* sub-menu items, the callback that gets executed when any of these are 
% clicked is the exactly the same one that corresponds to *Item One*. Similar behavior 
% is obtained with *Item Three* menu item. 

open_system('f14');

% Close all the scopes.
blocks = find_system(bdroot,'BlockType','Scope');
for i = 1:length(blocks)
set_param(blocks{i},'Open','off')
end

sl_refresh_customizations;

%%
% << sl_custom_demo_04.png>>
%
% If the "Red Screen" menu item is clicked, the background color
% screen toggles from white to red.
%
% <<sl_custom_demo_05.png>>

%% 
% We can revert back to the original GUI settings by using the following
% commands:

delete('sl_customization.m');
sl_refresh_customizations;
bdclose(gcs)

%%  Disabling and Hiding Options in Configuration Paramaters Dialog
% In this section we show how you can customize the *Configuration
% Parameters* dialog. First, we open the F14 model and show the various
% parameters in the dialog. Initially, the *Start Time*
% option is visible. The *Solver Type* and *Solver* are enabled.
% A close look at the |sl_customization.m| file shows that *Start Time* should be hidden.
% *Solver Type* and *Solver* should be disabled.

open_system('f14');

% Close all the scopes.
blocks = find_system(bdroot,'BlockType','Scope');
for i = 1:length(blocks)
set_param(blocks{i},'Open','off')
end

myConfigObj = getActiveConfigSet(gcs);
openDialog(myConfigObj);
copyfile('sl_custom_disable_hide.m', 'sl_customization.m','f');
type sl_customization.m

%%
% <<sl_custom_demo_06.png>>

%% 
% To see the dialog and widget names in the *Configuration Parameters*
% dialog, use the following commands:
%
% |cm = sl_customization_manager;|
%
% |cm.showWidgetIdAsToolTip = true;|
%
% The following command closes the *Configuration Parameters* dialog

closeDialog(myConfigObj);

%%
% The solver type is set to fixed-step discrete in the model's
% configuration set and the customizations are applied. The customization hides the *Start
% Time* dialog, sets the *Solver Type* to *Fixed-step*, sets *Solver* field to *Discrete*, and disables both options.
 
set_param(myConfigObj,'Solver','FixedStepDiscrete'); 
sl_refresh_customizations;
openDialog(myConfigObj);

%%
% <<sl_custom_demo_07.png>>

%% 
% We can revert back to the original Simulink GUI settings by using the following
% commands:

closeDialog(myConfigObj);
delete('sl_customization.m');
sl_refresh_customizations;
bdclose(gcs)
